<?php
namespace Src\Lib;

final class Uploads {

  public static function ensureReportDir(int $reportId): string {
    $dir = rtrim($GLOBALS['APP_CONFIG']['uploads']['dir'], '/');
    $reportDir = $dir . '/reports/' . $reportId;
    if (!is_dir($reportDir)) {
      @mkdir($reportDir, 0755, true);
    }
    return $reportDir;
  }

  public static function validateImageFile(array $file): void {
    $maxBytes = (int)$GLOBALS['APP_CONFIG']['uploads']['max_bytes_per_file'];
    $allowed = (array)$GLOBALS['APP_CONFIG']['uploads']['allowed_mime'];

    if (!isset($file['error']) || $file['error'] !== UPLOAD_ERR_OK) {
      Http::error(400, 'Upload failed.');
    }
    if (!isset($file['size']) || (int)$file['size'] <= 0) {
      Http::error(400, 'Empty upload.');
    }
    if ((int)$file['size'] > $maxBytes) {
      Http::error(400, 'File too large. Max 1MB.');
    }
    if (!isset($file['tmp_name']) || !is_uploaded_file($file['tmp_name'])) {
      Http::error(400, 'Invalid upload.');
    }

    $finfo = new \finfo(FILEINFO_MIME_TYPE);
    $mime = $finfo->file($file['tmp_name']) ?: '';
    if (!in_array($mime, $allowed, true)) {
      Http::error(400, 'Invalid file type. Only JPG/PNG allowed.');
    }
  }

  public static function moveToReportDir(int $reportId, array $file, int $index): array {
    self::validateImageFile($file);

    $reportDir = self::ensureReportDir($reportId);
    $ext = self::guessExtension($file);
    $name = 'photo_' . $index . '_' . bin2hex(random_bytes(6)) . '.' . $ext;

    $dest = $reportDir . '/' . $name;
    if (!move_uploaded_file($file['tmp_name'], $dest)) {
      Http::error(500, 'Failed to save file.');
    }

    $baseUrl = rtrim($GLOBALS['APP_CONFIG']['uploads']['base_url'], '/');
    $url = $baseUrl . '/reports/' . $reportId . '/' . $name;

    return ['path' => $dest, 'url' => $url];
  }

  private static function guessExtension(array $file): string {
    $finfo = new \finfo(FILEINFO_MIME_TYPE);
    $mime = $finfo->file($file['tmp_name']) ?: '';
    if ($mime === 'image/png') return 'png';
    return 'jpg';
  }
}
