<?php
namespace Src\Lib;

use DateTimeImmutable;

final class RateLimit {

  public static function enforce(string $action, int $maxPerDay): void {
    $pdo = Db::pdo();
    $ipHash = Security::hashIdentifier(Security::getClientIp());
    $deviceHash = Security::hashIdentifier(Security::getDeviceId());

    $key = Security::sha256($action . '|' . $ipHash . '|' . $deviceHash);
    $today = (new DateTimeImmutable('now'))->format('Y-m-d');
    $windowStart = $today . ' 00:00:00';

    $stmt = $pdo->prepare('SELECT window_start, count FROM rate_limits WHERE key_hash = ? LIMIT 1');
    $stmt->execute([$key]);
    $row = $stmt->fetch();

    if (!$row) {
      $ins = $pdo->prepare('INSERT INTO rate_limits (key_hash, window_start, count) VALUES (?, ?, 1)');
      $ins->execute([$key, $windowStart]);
      return;
    }

    if (substr($row['window_start'], 0, 10) !== $today) {
      $upd = $pdo->prepare('UPDATE rate_limits SET window_start = ?, count = 1 WHERE key_hash = ?');
      $upd->execute([$windowStart, $key]);
      return;
    }

    $count = (int)$row['count'];
    if ($count >= $maxPerDay) {
      Http::error(429, 'Rate limit exceeded. Try again later.');
    }

    $upd = $pdo->prepare('UPDATE rate_limits SET count = count + 1 WHERE key_hash = ?');
    $upd->execute([$key]);
  }
}
