<?php
namespace Src\Controllers;

use Src\Lib\Db;
use Src\Lib\Http;
use Src\Lib\RateLimit;
use Src\Lib\Security;
use Src\Lib\Uploads;

final class ReportsController {

  public function createReport(): void {
    $limits = $GLOBALS['APP_CONFIG']['rate_limit'];
    RateLimit::enforce('create_report', (int)$limits['reports_per_day']);

    $body = Http::jsonBody();
    $county = trim((string)($body['county'] ?? ''));
    $type = trim((string)($body['type'] ?? ''));
    $desc = trim((string)($body['description'] ?? ''));
    $approxLabel = trim((string)($body['approxLocationLabel'] ?? ''));
    $lat = $body['latApprox'] ?? null;
    $lng = $body['lngApprox'] ?? null;

    if ($county === '' || !in_array($county, ['Nyeri', 'Kirinyaga'], true)) Http::error(400, 'Invalid county.');
    if ($type === '' || strlen($type) > 50) Http::error(400, 'Invalid type.');
    if (mb_strlen($desc) < 10) Http::error(400, 'Description too short.');
    if (mb_strlen($approxLabel) < 3 || mb_strlen($approxLabel) > 120) Http::error(400, 'Approx location label required.');
    if (!is_numeric($lat) || !is_numeric($lng)) Http::error(400, 'latApprox/lngApprox required.');

    $lat = round((float)$lat, 3);
    $lng = round((float)$lng, 3);

    $pdo = Db::pdo();
    $stmt = $pdo->prepare('
      INSERT INTO reports
      (county, type, description, approx_label, lat_approx, lng_approx, photo_count, status, created_at, device_id_hash, ip_hash)
      VALUES (?, ?, ?, ?, ?, ?, 0, "PUBLISHED", NOW(), ?, ?)
    ');

    $deviceHash = Security::hashIdentifier(Security::getDeviceId());
    $ipHash = Security::hashIdentifier(Security::getClientIp());

    $stmt->execute([$county, $type, $desc, $approxLabel, $lat, $lng, $deviceHash, $ipHash]);
    $reportId = (int)$pdo->lastInsertId();

    Http::created(['reportId' => $reportId, 'uploadPhotosUrl' => '/v1/reports/' . $reportId . '/photos']);
  }

  public function uploadPhotos(int $reportId): void {
    $limits = $GLOBALS['APP_CONFIG']['rate_limit'];
    RateLimit::enforce('upload_photos', (int)$limits['photo_uploads_per_day']);

    $pdo = Db::pdo();
    $s = $pdo->prepare('SELECT id, status, photo_count FROM reports WHERE id = ? LIMIT 1');
    $s->execute([$reportId]);
    $r = $s->fetch();
    if (!$r) Http::error(404, 'Report not found.');
    if ($r['status'] !== 'PUBLISHED' && $r['status'] !== 'HIDDEN') Http::error(400, 'Report cannot accept uploads.');

    $maxFiles = (int)$GLOBALS['APP_CONFIG']['uploads']['max_files'];
    $existing = (int)$r['photo_count'];
    if ($existing >= $maxFiles) Http::error(400, 'This report already has max photos.');

    if (!isset($_FILES['photos'])) Http::error(400, 'Missing photos[]. Use multipart field name "photos".');

    $files = $_FILES['photos'];
    $count = is_array($files['name']) ? count($files['name']) : 0;
    if ($count <= 0) Http::error(400, 'No files received.');

    $remaining = $maxFiles - $existing;
    $toProcess = min($remaining, $count);

    $photoUrls = [];
    for ($i = 0; $i < $toProcess; $i++) {
      $file = [
        'name' => $files['name'][$i],
        'type' => $files['type'][$i],
        'tmp_name' => $files['tmp_name'][$i],
        'error' => $files['error'][$i],
        'size' => $files['size'][$i],
      ];

      $index = $existing + $i + 1;
      $saved = Uploads::moveToReportDir($reportId, $file, $index);

      $ins = $pdo->prepare('INSERT INTO report_photos (report_id, url, created_at) VALUES (?, ?, NOW())');
      $ins->execute([$reportId, $saved['url']]);
      $photoUrls[] = $saved['url'];
    }

    $newCount = $existing + count($photoUrls);
    $upd = $pdo->prepare('UPDATE reports SET photo_count = ? WHERE id = ?');
    $upd->execute([$newCount, $reportId]);

    Http::ok(['reportId' => $reportId, 'photoCount' => $newCount, 'photoUrlsAdded' => $photoUrls]);
  }

  public function listReports(): void {
    $county = trim((string)($_GET['county'] ?? ''));
    $limit = (int)($_GET['limit'] ?? 100);
    $limit = max(1, min(300, $limit));

    $pdo = Db::pdo();

    if ($county !== '') {
      if (!in_array($county, ['Nyeri', 'Kirinyaga'], true)) Http::error(400, 'Invalid county.');
      $stmt = $pdo->prepare('
        SELECT r.*,
          (SELECT JSON_ARRAYAGG(p.url) FROM report_photos p WHERE p.report_id = r.id) AS photo_urls
        FROM reports r
        WHERE r.status = "PUBLISHED" AND r.county = ?
        ORDER BY r.created_at DESC
        LIMIT ?
      ');
      $stmt->bindValue(1, $county);
      $stmt->bindValue(2, $limit, \PDO::PARAM_INT);
      $stmt->execute();
    } else {
      $stmt = $pdo->prepare('
        SELECT r.*,
          (SELECT JSON_ARRAYAGG(p.url) FROM report_photos p WHERE p.report_id = r.id) AS photo_urls
        FROM reports r
        WHERE r.status = "PUBLISHED"
        ORDER BY r.created_at DESC
        LIMIT ?
      ');
      $stmt->bindValue(1, $limit, \PDO::PARAM_INT);
      $stmt->execute();
    }

    $rows = $stmt->fetchAll();
    $out = [];
    foreach ($rows as $row) {
      $photos = [];
      if (!empty($row['photo_urls'])) {
        $decoded = json_decode($row['photo_urls'], true);
        if (is_array($decoded)) $photos = $decoded;
      }
      $out[] = [
        'id' => (int)$row['id'],
        'county' => $row['county'],
        'type' => $row['type'],
        'description' => $row['description'],
        'approxLocationLabel' => $row['approx_label'],
        'latApprox' => (float)$row['lat_approx'],
        'lngApprox' => (float)$row['lng_approx'],
        'photoCount' => (int)$row['photo_count'],
        'photoUrls' => $photos,
        'createdAt' => $row['created_at'],
      ];
    }

    Http::ok(['reports' => $out]);
  }

  public function flagReport(int $reportId): void {
    $limits = $GLOBALS['APP_CONFIG']['rate_limit'];
    RateLimit::enforce('flag_report', (int)$limits['flags_per_day']);

    $body = Http::jsonBody();
    $reason = trim((string)($body['reason'] ?? ''));
    if ($reason === '' || mb_strlen($reason) > 200) Http::error(400, 'Reason required (max 200 chars).');

    $pdo = Db::pdo();
    $s = $pdo->prepare('SELECT id FROM reports WHERE id = ? LIMIT 1');
    $s->execute([$reportId]);
    if (!$s->fetch()) Http::error(404, 'Report not found.');

    $deviceHash = Security::hashIdentifier(Security::getDeviceId());
    $ipHash = Security::hashIdentifier(Security::getClientIp());

    $ins = $pdo->prepare('INSERT INTO flags (report_id, reason, created_at, device_id_hash, ip_hash) VALUES (?, ?, NOW(), ?, ?)');
    $ins->execute([$reportId, $reason, $deviceHash, $ipHash]);

    Http::ok(['flagged' => true]);
  }
}
