<?php
namespace Src\Controllers;

use Src\Lib\Db;
use Src\Lib\Http;
use Src\Lib\Security;

final class AdminController {

  public function login(): void {
    $body = Http::jsonBody();
    $email = strtolower(trim((string)($body['email'] ?? '')));
    $password = (string)($body['password'] ?? '');
    if ($email === '' || $password === '') Http::error(400, 'Email and password required.');

    $pdo = Db::pdo();
    $s = $pdo->prepare('SELECT id, password_hash FROM admins WHERE email = ? LIMIT 1');
    $s->execute([$email]);
    $admin = $s->fetch();
    if (!$admin || !password_verify($password, $admin['password_hash'])) Http::error(401, 'Invalid credentials.');

    $adminId = (int)$admin['id'];
    $ttl = (int)$GLOBALS['APP_CONFIG']['admin_token_ttl'];
    $exp = time() + $ttl;

    $payload = $adminId . '|' . $exp . '|' . bin2hex(random_bytes(12));
    $sig = Security::signToken($payload);
    $token = base64_encode($payload . '|' . $sig);

    $ins = $pdo->prepare('INSERT INTO admin_tokens (admin_id, token, expires_at, created_at) VALUES (?, ?, FROM_UNIXTIME(?), NOW())');
    $ins->execute([$adminId, $token, $exp]);

    Http::ok(['token' => $token, 'expiresAt' => $exp]);
  }

  public function moderate(int $reportId, string $action): void {
    $adminId = $this->requireAdmin();
    $pdo = Db::pdo();

    $status = $action === 'hide' ? 'HIDDEN' : 'REMOVED';

    $s = $pdo->prepare('SELECT id FROM reports WHERE id = ? LIMIT 1');
    $s->execute([$reportId]);
    if (!$s->fetch()) Http::error(404, 'Report not found.');

    $upd = $pdo->prepare('UPDATE reports SET status = ? WHERE id = ?');
    $upd->execute([$status, $reportId]);

    Http::ok(['reportId' => $reportId, 'status' => $status, 'adminId' => $adminId]);
  }

  private function requireAdmin(): int {
    $token = Security::bearerToken();
    if ($token === '') Http::error(401, 'Missing bearer token.');

    $pdo = Db::pdo();
    $s = $pdo->prepare('SELECT admin_id, expires_at FROM admin_tokens WHERE token = ? LIMIT 1');
    $s->execute([$token]);
    $row = $s->fetch();
    if (!$row) Http::error(401, 'Invalid token.');

    $expires = strtotime($row['expires_at']);
    if ($expires !== false && $expires < time()) Http::error(401, 'Token expired.');

    return (int)$row['admin_id'];
  }
}
